'use strict'

entityRegistry['module']['rising'] = {
    extendedInfo: {
        displayName: 'Rising',
        displayGroup: '3D Effects',
    },
    init: (staticConfig) => {
        const {
            numItems,
            cavitySize,
            spawnSize,
            spawnHeight,
            ballMass,
            preRoll,
            frameCount,
            seed,
        } = { ...staticConfig }
        const rng = new Math.seedrandom(seed)

        const world = new CANNON.World()
        world.gravity.set(0, 0, -9.82) // m/s²

        // Materials
        var groundMaterial = new CANNON.Material('groundMaterial')
        // Adjust constraint equation parameters for ground/ground contact
        var ground_ground_cm = new CANNON.ContactMaterial(groundMaterial, groundMaterial, {
            friction: .8,
            restitution: 0.3,
            contactEquationStiffness: 1e8,
            contactEquationRelaxation: 3,
            frictionEquationStiffness: 1e8,
            frictionEquationRegularizationTime: 3,
        });
        // Add contact material to the world
        world.addContactMaterial(ground_ground_cm)

        // // Create a slippery material (friction coefficient = 0.0)
        // var slipperyMaterial = new CANNON.Material("slipperyMaterial");
        // // The ContactMaterial defines what happens when two materials meet.
        // // In this case we want friction coefficient = 0.0 when the slippery material touches ground.
        // var slippery_ground_cm = new CANNON.ContactMaterial(groundMaterial, slipperyMaterial, {
        //     friction: 0,
        //     restitution: 0.3,
        //     contactEquationStiffness: 1e8,
        //     contactEquationRelaxation: 3
        // });
        // // We must add the contact materials to the world
        // world.addContactMaterial(slippery_ground_cm);

        const sphereBody = []
        const sphere = new CANNON.Sphere(1)
        for (let i = 0; i < numItems; ++i) {
            const body = new CANNON.Body({
                mass: ballMass,
                position: new CANNON.Vec3((rng()-.5)*spawnSize[0], (rng()-.5)*spawnSize[1], spawnHeight + (rng()-.5)*spawnSize[2]),
                quaternion: new CANNON.Quaternion(rng(), rng(), rng(), rng()),
                shape: sphere,
                material: groundMaterial
            })
            world.addBody(body)
            // body.applyLocalForce(new CANNON.Vec3(rng()*2-1, rng()*2-1, -(rng()*5+5)), new CANNON.Vec3(rng()*2-1, rng()*2-1, rng()*2-1))
            sphereBody.push(body)
        }

        // Floor
        world.addBody(new CANNON.Body({
            mass: 0, // mass == 0 makes the body static
            position: new CANNON.Vec3(0, 0, -cavitySize[2]),
            shape: new CANNON.Box(new CANNON.Vec3(500,500,1)),
            material: groundMaterial,
        }))
        world.addBody(new CANNON.Body({
            mass: 0,
            position: new CANNON.Vec3(0, -cavitySize[1], 0),
            shape: new CANNON.Box(new CANNON.Vec3(500,1,500)),
            material: groundMaterial,
        }))
        world.addBody(new CANNON.Body({
            mass: 0,
            position: new CANNON.Vec3(0, cavitySize[1], 0, 0),
            shape: new CANNON.Box(new CANNON.Vec3(500,1,500)),
            material: groundMaterial,
        }))
        world.addBody(new CANNON.Body({
            mass: 0,
            position: new CANNON.Vec3(-cavitySize[0], 0, 0, 0),
            shape: new CANNON.Box(new CANNON.Vec3(1,500,500)),
            material: groundMaterial,
        }))
        world.addBody(new CANNON.Body({
            mass: 0,
            position: new CANNON.Vec3(cavitySize[0], 0, 0, 0),
            shape: new CANNON.Box(new CANNON.Vec3(1,500,500)),
            material: groundMaterial,
        }))

        var fixedTimeStep = 1.0 / 100.0 // seconds

        for (let i = 0; i < preRoll; ++i) {
            world.step(fixedTimeStep)
        }

        const bodies = []
        for (let i = 0; i < frameCount; ++i) {
            for (let j = 0; j < sphereBody.length; ++j) {
                bodies[j] = bodies[j] || []
                const pos = [sphereBody[j].position.x, -sphereBody[j].position.z, sphereBody[j].position.y]
                const quat = sphereBody[j].quaternion
                // const rot = m4.quaternionToEuler([-quat.y, quat.z, -quat.x, quat.w])
                // bodies[j].push([pos, [-rot[0], -rot[1], -rot[2]], mat])
                const mat = m4.quaternionToMatrix([quat.y, quat.z, quat.x, quat.w])
                bodies[j].push([pos, mat])
            }
            world.step(fixedTimeStep)
        }

        return {
            bodies
        }
    },
    staticConfig: [
        { paramName: 'numItems', displayName: 'Items', type: 'int', defaultValue: 100, triggerInit: true },
        { paramName: 'cavitySize', displayName: 'Cavity Size', type: 'float3', defaultValue: [28, 3, 20], triggerInit: true },
        { paramName: 'spawnSize', displayName: 'Spawn Size', type: 'float3', defaultValue: [50, 300, 50], triggerInit: true },
        { paramName: 'spawnHeight', displayName: 'Spawn Height', type: 'float', defaultValue: 300, triggerInit: true },
        { paramName: 'ballMass', displayName: 'Ball Mass', type: 'float', defaultValue: 5.25, triggerInit: true },
        { paramName: 'preRoll', displayName: 'Pre Roll', type: 'int', defaultValue: 0, triggerInit: true },
        { paramName: 'frameCount', displayName: 'Frame Count', type: 'int', defaultValue: 1000, triggerInit: true },
        { paramName: 'seed', displayName: 'Seed', type: 'string', defaultValue: 'seed', triggerInit: true },
    ],
    dynamicConfig: [
        { paramName: 'model', displayName: 'Model', type: 'model', defaultValue: '' },
        { paramName: 'itemScale', displayName: 'Item Scale', type: 'float', defaultValue: 1 },
        { paramName: 'animation', displayName: 'Animation', type: 'float', defaultValue: 0 },
    ],
    actions: {
        'render': (self, frameTime, config, ctx) => {
            const {
                frameCount,
                model,
                itemScale,
                animation,
            } = { ...config }

            const colorBuffer = renderer.getCurrentBuffer('color')
            const depthBuffer = renderer.getCurrentBuffer('depth')
            const brightnessBuffer = renderer.getCurrentBuffer('brightness')

            const scaleMat = m4.scaling(itemScale, itemScale, itemScale)
            const clampedAnimation = clamp(animation, 0, 1)
            const frame = Math.floor(clampedAnimation * (frameCount-1))
            const frameAlpha = (clampedAnimation * (frameCount-1)) % 1
            self.bodies.forEach(body => {
                const pos1 = body[frame][0]
                const pos2 = body[Math.min(frameCount-1, frame+1)][0]
                const pos = m4.lerpVectors(pos1, pos2, frameAlpha)
                if (pos[1] > -50 && pos[1] < 20) {
                    const rotationMat = body[frame][1]

                    // const rot1 = body[frame][1]
                    // const rot2 = body[frame+1][1]
                    // const rot = [
                    //     Math.abs(rot1[0]-rot2[0])>Math.PI ? rot1[0] : lerp(rot1[0],rot2[0],frameAlpha),
                    //     Math.abs(rot1[1]-rot2[1])>Math.PI ? rot1[1] : lerp(rot1[1],rot2[1],frameAlpha),
                    //     Math.abs(rot1[2]-rot2[2])>Math.PI ? rot1[2] : lerp(rot1[2],rot2[2],frameAlpha),
                    // ]
                    // const rotationMat = m4.zRotate(m4.yRotate(m4.xRotation(rot[0]), rot[1]), rot[2])

                    const translationMat = m4.translation(pos[0], pos[1], pos[2])
                    const worldMat = m4.multiply(m4.multiply(translationMat, rotationMat), scaleMat)

                    renderer.drawModel(model, worldMat, colorBuffer, depthBuffer, brightnessBuffer)
                }
            })
        }
    }
}
